//
// keys.cpp
//
// ITfKeyEventSink implementation.
//

#include "Log.h"
#include "globals.h"
#include "mark.h"
#include "editsess.h"
#include <vector>
#include <chrono>
#include "KeystrokeEditSession.h"
#include "RetrySession.h"

using namespace std;
using namespace std::chrono; 

//+---------------------------------------------------------------------------
//
// _FinishInput
//
// Returns S_OK to eat the keystroke, S_FALSE otherwise.
//----------------------------------------------------------------------------

HRESULT CMarkTextService::_FinishInput(TfEditCookie ec, ITfContext *pContext)
{
	// just terminate the composition
	_TerminateComposition(ec);
	LogOutput(__FUNCTION__, __LINE__, "Convert End!!");

	return S_OK;
}

//+---------------------------------------------------------------------------
//
// _HandleNormalization
//
// Returns S_OK to eat the keystroke, S_FALSE otherwise.
//----------------------------------------------------------------------------

HRESULT CMarkTextService::_HandleNormalization(TfEditCookie ec, ITfContext *pContext)
{
	std::wstring words = compositor_.Normalize();
	if (words.size() == 0) {
		return S_OK;
	}

	if (SetTextToComposition(ec, pContext, _pComposition, words) != S_OK) {
		LogOutput(__FUNCTION__, __LINE__, "Failed To SetTextToComposition");
		return S_FALSE;
	}

	return S_OK;
}

//+---------------------------------------------------------------------------
//
// _HandleAsciiInput
//
// Returns S_OK to eat the keystroke, S_FALSE otherwise.
//----------------------------------------------------------------------------

HRESULT CMarkTextService::_HandleAsciiInput(TfEditCookie ec, ITfContext *pContext)
{
	std::wstring wstrKeyInput = compositor_.GetInputKeys();
	if (wstrKeyInput.size() == 0) {
		wstrKeyInput = L"\n";
	}
	if (SetTextToComposition(ec, pContext, _pComposition, wstrKeyInput) != S_OK) {
		LogOutput(__FUNCTION__, __LINE__, "Failed To SetTextToComposition");
		return S_FALSE;
	}

	return S_OK;
}

//+---------------------------------------------------------------------------
//
// _HandleBack
//
// Returns S_OK to eat the keystroke, S_FALSE otherwise.
//----------------------------------------------------------------------------

HRESULT CMarkTextService::_HandleBack(TfEditCookie ec, ITfContext *pContext)
{
	wstring textInput;
	wstring textWords;
	// ͕Aϊ폜
	compositor_.Delete(textInput, textWords);

	if (SetTextToComposition(ec, pContext, _pComposition, textInput) != S_OK) {
		LogOutput(__FUNCTION__, __LINE__, "Failed To SetTextToComposition");
		return S_FALSE;
	}
	this->_SetCompositionDisplayAttributes(ec);

	// ϊ\
	auto prc = cursorManager_.GetPRC(ec, _pComposition);
	wndInputText->SetText(textWords, prc);

	if (textInput.size() == 0) {
		_FinishInput(ec, pContext);
	}
	return S_OK;
}

//+---------------------------------------------------------------------------
//
// _HandleKeyDown
//
// If the keystroke happens within a composition, eat the key and return S_OK.
// Otherwise, do nothing and return S_FALSE.
//----------------------------------------------------------------------------

HRESULT CMarkTextService::_HandleKeyDownConvert(TfEditCookie ec, ITfContext *pContext, WPARAM wParam, BOOL with_shift)
{
	TF_SELECTION tfSelection;
	ULONG cFetched;
	HRESULT hr;
	// SHIFTL[̑Ō̎͂ŏI
	if (wParam == VK_SHIFT)
	{
		return S_FALSE;
	}

	// zL[R[hASCIIR[hɕϊ(L[{[hłǂ̕ꂽ̂F)
	WCHAR ch = keyboadManager_->ComposeChar(wParam, with_shift);

	// ΉL[ꂽƂ
	if (ch == 0x00) {
		return S_FALSE;
	}

	// ϊΏۂ̓̓L[ǉ
	if (!compositor_.AddKey(ch)) {
		// ͂ꂽL[ϊe[uɑ݂ȂƂ͖
		return S_FALSE;
	}
	
	// EChE̕\
	wndInputText->Show(TRUE);

	// first, test where a keystroke would go in the document if we did an insert
	if (pContext->GetSelection(ec, TF_DEFAULT_SELECTION, 1, &tfSelection, &cFetched) != S_OK || cFetched != 1) {
        return S_FALSE;
	}

	// J[\E[ɍ킹
	AdjustRangeToComposition(ec, tfSelection.range, _pComposition, false);
	tfSelection.range->Collapse(ec, TF_ANCHOR_END);

	hr = S_OK; // return S_FALSE to NOT eat the key

	auto input_keys = compositor_.GetInputKeys();

	// insert the text
	if (tfSelection.range->SetText(ec, 0, &ch, 1) != S_OK) {
		goto Exit;
	}
	tfSelection.range->Collapse(ec, TF_ANCHOR_END);

    pContext->SetSelection(ec, 1, &tfSelection);

    // apply our dislay attribute property to the inserted text
    // we need to apply it to the entire composition, since the
    // display attribute property is static, not static compact
    _SetCompositionDisplayAttributes(ec);

Exit:
	tfSelection.range->Release();
	return hr;
}

//+---------------------------------------------------------------------------
//
// _InitKeystrokeSink
//
// Advise a keystroke sink.
//----------------------------------------------------------------------------

BOOL CMarkTextService::_InitKeystrokeSink()
{
    ITfKeystrokeMgr *pKeystrokeMgr;
    HRESULT hr;

    if (_pThreadMgr->QueryInterface(IID_ITfKeystrokeMgr, (void **)&pKeystrokeMgr) != S_OK)
        return FALSE;

    hr = pKeystrokeMgr->AdviseKeyEventSink(_tfClientId, (ITfKeyEventSink *)this, TRUE);

    pKeystrokeMgr->Release();

    return (hr == S_OK);
}

//+---------------------------------------------------------------------------
//
// _UninitKeystrokeSink
//
// Unadvise a keystroke sink.  Assumes we have advised one already.
//----------------------------------------------------------------------------

void CMarkTextService::_UninitKeystrokeSink()
{
    ITfKeystrokeMgr *pKeystrokeMgr;

    if (_pThreadMgr->QueryInterface(IID_ITfKeystrokeMgr, (void **)&pKeystrokeMgr) != S_OK)
        return;

    pKeystrokeMgr->UnadviseKeyEventSink(_tfClientId);

    pKeystrokeMgr->Release();
}

//+---------------------------------------------------------------------------
//
// _InitLayoutSink
//
// Advise a layout sink.
//----------------------------------------------------------------------------

BOOL CMarkTextService::_InitLayoutSink(ITfDocumentMgr *pDocMgr)
{
	ITfSource* pSource;

    if (_dwTextLayoutSinkCookie != TF_INVALID_COOKIE)
    {
        if (_pTextLayoutSinkContext->QueryInterface(IID_ITfSource, (void **)&pSource) == S_OK)
        {
            pSource->UnadviseSink(_dwTextLayoutSinkCookie);
            pSource->Release();
        }

        _pTextLayoutSinkContext->Release();
        _pTextLayoutSinkContext = NULL;
        _dwTextLayoutSinkCookie = TF_INVALID_COOKIE;
    }

	if (pDocMgr == NULL) {
        return TRUE;
	}

	if (pDocMgr->GetTop(&_pTextLayoutSinkContext) != S_OK) {
        return FALSE;
	}

	BOOL success = AdviseSink(_pTextLayoutSinkContext, (ITfTextLayoutSink *) this, IID_ITfTextLayoutSink, &_dwTextLayoutSinkCookie);
	if (!success) {
		_pTextLayoutSinkContext->Release();
		_pTextLayoutSinkContext = NULL;
	}
	return success;
}

//+---------------------------------------------------------------------------
//
// OnSetFocus
//
// Called by the system whenever this service gets the keystroke device focus.
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnSetFocus(BOOL fForeground)
{
//	LogOutput(__FUNCTION__, __LINE__, " fForeground[%d] _IsComposing[%d]", fForeground, _IsComposing());
	return S_OK;
}


BOOL CMarkTextService::DeterminEat(WPARAM wParam, ITfContext *pContext) {
	switch (wParam)
	{
	case VK_SPACE:
	case VK_BACK:
	case VK_DELETE:
	case VK_TAB:
	case VK_RETURN:
		return compositor_.WordLength() != 0;
		break;
	default:
		// CtrlL[ĂƂ̓V[gJbgL[Ƃ݂Ȃϊ͍sȂ
		auto with_ctrl = (GetKeyState(VK_CONTROL) & 0x8000) != 0;
		if (with_ctrl) {
			return FALSE;
		}
		else {
			return keyboadManager_->Available(wParam);
		}
		break;
	}
}

BOOL CMarkTextService::IsArrowKey(WPARAM wParam) {
	switch (wParam)
	{
	case VK_LEFT:
	case VK_RIGHT:
	case VK_UP:
	case VK_DOWN:
	 	return TRUE;
	default:
		return FALSE;
	}
}

BOOL CMarkTextService::WithShift() {
	return (GetKeyState(VK_SHIFT) & 0x8000) > 0;
}

//+---------------------------------------------------------------------------
//
// OnTestKeyDown
//
// Called by the system to query if this service wants a potential keystroke.
//
// :
// UWPAv3DyCgŎł͌ĂяoȂB
// UWPAvNotepad++ł͎OɌĂяoA*pfEatenFALSÊƂAOnKeyDown͌ĂяoȂB
// Notepad++ɂẮAOnTestKeyDownpfEaten == TRUEOnKeyDownpfEaten == FALSÊƂAOnKeyDownpfEaten͍lȂ߁AOnKeyTestDown̎_ŒeĂKvB
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnTestKeyDown(ITfContext *pContext, WPARAM wParam, LPARAM lParam, BOOL *pfEaten)
{
	*pfEaten = DeterminEat(wParam, pContext);
	if (!*pfEaten) {
		if (IsArrowKey(wParam)){
			if (cursorManager_.RestartRequired() && _IsComposing()) {
				// ExcelȂǈꕔ̃AvComposition܂܏\L[abort邽ߍċNB
				RestartInput(pContext);
			}
		}
		return S_OK;
	}
	return S_OK;
}

//+---------------------------------------------------------------------------
//
// OnKeyDown
//
// Called by the system to offer this service a keystroke.  If *pfEaten == TRUE
// on exit, the application will not handle the keystroke.
//
// This text service is interested in handling keystrokes to demonstrate the
// use the compositions.  Some apps will cancel compositions if they receive
// keystrokes while a compositions is ongoing.
//----------------------------------------------------------------------------
STDAPI CMarkTextService::OnKeyDown(ITfContext *pContext, WPARAM wParam, LPARAM lParam, BOOL *pfEaten)
{
	// OnTestKeyDownĂяoȂƂ̂߂ɂłO肷B
	*pfEaten = DeterminEat(wParam, pContext);
	if (!*pfEaten) {
		if (IsArrowKey(wParam)){
			if (cursorManager_.RestartRequired() && _IsComposing()) {
				// ExcelȂǈꕔ̃AvComposition܂܏\L[abort邽ߍċNB
				RestartInput(pContext);
			}
		}
		return S_OK;
	}else if (wParam == VK_TAB) {
		return S_OK;
	}
	if (_pComposition == NULL) {
		SetComposition(pContext);
	}

    CKeystrokeEditSession *pEditSession;
    HRESULT hr;

    hr = E_FAIL;
    *pfEaten = FALSE;

    if (_pComposition != NULL) // only eat keys while composing
    {
        // we'll insert a char ourselves in place of this keystroke
        if ((pEditSession = new CKeystrokeEditSession(this, pContext, wParam, WithShift())) == NULL)
            goto Exit;

		// we need a lock to do our work
        // nb: this method is one of the few places where it is legal to use
        // the TF_ES_SYNC flag
        if (pContext->RequestEditSession(_tfClientId, pEditSession, TF_ES_SYNC | TF_ES_READWRITE, &hr) != S_OK)
        {
            hr = E_FAIL;
        }
		pEditSession->Release();
    }

Exit:
    // if we made it all the way to the RequestEditSession, then hr is ultimately the
    // return code from CKeystrokeEditSession::DoEditSession.  Our DoEditSession method
    // return S_OK to signal that the keystroke should be eaten, S_FALSE otherwise.
    if (hr == S_OK)
    {
		LogOutput(__FUNCTION__, __LINE__, "akkharaComposition.GetWordLen()[%d] _pComposition=%p", compositor_.WordLength(), _pComposition);
		switch (wParam)
		{
		case VK_SPACE:
			// m蕶񂪂ȂԂł͂̂܂܉L[̋@\
			if (compositor_.WordLength() == 0)
			{
				*pfEaten = FALSE;
			}
			else if (_pComposition != NULL) {
				// ͊mɂϊI
				*pfEaten = TRUE;
			}
			else {
				// ͎sȂ͂
				*pfEaten = FALSE;
				LogOutput(__FUNCTION__, __LINE__, "WARNING: _pComposition is NULL, suspecting unintended release");
			}
			RestartInput(pContext);
			break;
		case VK_BACK:
		case VK_DELETE:
			// m蕶񂪂ȂԂł͂̂܂܉L[̋@\
			if (compositor_.WordLength() == 0)
			{
				*pfEaten = FALSE;
				RestartInput(pContext);
			}
			else {
				*pfEaten = _pComposition != NULL;
			}
			break;
		case VK_RETURN:
			*pfEaten = _pComposition != NULL;
			RestartInput(pContext);
			break;
		default:
			*pfEaten = TRUE;
		}
    }
	LogOutput(__FUNCTION__, __LINE__, "Key Code[0x%04x] Eaten[%d] hr[%d]", wParam, *pfEaten, hr);
	LogOutput("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n");
	cursorManager_.ResetPRC(false);
	return S_OK;
}

BOOL CMarkTextService::IsInvalidComposition(ITfContext *pContext, TfEditCookie ec) {
	TF_SELECTION tfSelection;
	ULONG cFetched;
	// first, test where a keystroke would go in the document if we did an insert
	if (pContext->GetSelection(ec, TF_DEFAULT_SELECTION, 1, &tfSelection, &cFetched) != S_OK || cFetched != 1) {
        return TRUE;
	}

    // is the insertion point covered by a composition?
	ITfRange *pRangeComposition;
	BOOL decision = FALSE;
    if (_pComposition->GetRange(&pRangeComposition) == S_OK)
    {
        BOOL fCovered = IsRangeCovered(ec, tfSelection.range, pRangeComposition);
		pRangeComposition->Release();
		decision = !fCovered;
    }
	tfSelection.range->Release();
	return decision;
}

//+---------------------------------------------------------------------------
//
// OnTestKeyUp
//
// Called by the system to query this service wants a potential keystroke.
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnTestKeyUp(ITfContext *pContext, WPARAM wParam, LPARAM lParam, BOOL *pfEaten)
{
//	LogOutput(__FUNCTION__, __LINE__, "Key Code[0x%04x][0x%04x] Eaten[%d] _IsComposing[%d]", wParam, lParam, *pfEaten, _IsComposing());
	*pfEaten = FALSE;
    return S_OK;
}

//+---------------------------------------------------------------------------
//
// OnKeyUp
//
// Called by the system to offer this service a keystroke.  If *pfEaten == TRUE
// on exit, the application will not handle the keystroke.
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnKeyUp(ITfContext *pContext, WPARAM wParam, LPARAM lParam, BOOL *pfEaten)
{
//	LogOutput(__FUNCTION__, __LINE__, "Key Code[0x%04x][0x%04x] Eaten[%d] _IsComposing[%d]", wParam, lParam, *pfEaten, _IsComposing());
	*pfEaten = FALSE;
	return S_OK;
}

//+---------------------------------------------------------------------------
//
// OnPreservedKey
//
// Called when a hotkey (registered by us, or by the system) is typed.
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnPreservedKey(ITfContext *pContext, REFGUID rguid, BOOL *pfEaten)
{
    *pfEaten = FALSE;
    return S_OK;
}


//+---------------------------------------------------------------------------
//
// HandleRetry
//
//----------------------------------------------------------------------------
HRESULT CMarkTextService::_HandleRetry(TfEditCookie ec, ITfContext* pContext, ITfContextView* pContextView) 
{
	RECT prc;
	if (_IsComposing()) {
		prc = cursorManager_.GetPRC(ec, _pComposition);
	}
	else {
		prc = cursorManager_.GetPRC(ec, pContext, pContextView);
	}
	wndInputText->RetryShowing(prc);

	if (!_IsComposing() && compositor_.GetInputKeys().size() > 0) {
		SetComposition(pContext);
		SetTextToComposition(ec, pContext, _pComposition, compositor_.GetInputKeys());
		this->_SetCompositionDisplayAttributes(ec);
	}

	return S_OK;
}

//+---------------------------------------------------------------------------
//
// OnLayoutChange
// 
// uEUȂǁÃvZXȂAvP[Vł́AeLXg͎ɃJ[\ʒułȂB
// ̏ꍇAJ[\͈ʒułĂOnLayoutChangeĂяoƂ̂ƁB
// āAOnLayoutChangeĂяoĂ߂ĕϊ\KvB
// Ql: https://d-toybox.com/studio/weblog/show.php?id=2018021600
//----------------------------------------------------------------------------
STDMETHODIMP CMarkTextService::OnLayoutChange(ITfContext* pContext, TfLayoutCode layoutCode, ITfContextView* pContextView) 
{
	ITfEditSession *pSession = NULL;
	if ((pSession = new CRetrySession(this, pContext, pContextView)) == NULL) {
		return E_FAIL;
	}

	HRESULT hr;
	if (pContext->RequestEditSession(_tfClientId, pSession, TF_ES_ASYNC | TF_ES_READWRITE, &hr) != S_OK)
	{
		pSession->Release();
		return E_FAIL;
	}
	pSession->Release();

	return S_OK;
}

HRESULT CMarkTextService::_HandleSpace(TfEditCookie ec, ITfContext *pContext)
{
	// ϊ擾
	wstring words = compositor_.Convert();
	if (words.size() != 0) {
		this->_SetCompositionDisplayAttributes(ec);

		// ϊ̓eLXgEChEɕ\
		auto prc = cursorManager_.GetPRC(ec, _pComposition);
		wndInputText->SetText(words, prc);
	}

	return S_OK;
}

// Cvbg[hċN
void CMarkTextService::RestartInput(ITfContext *pContext)
{
	wndInputText->Show(FALSE);
	cursorManager_.ResetPRC(true);
	compositor_.Reset();
	// just terminate the composition
	_TerminateCompositionInContext(pContext);
	LogOutput(__FUNCTION__, __LINE__, "Convert End!!");
}

// ʃEChE̕NA
void CMarkTextService::WorkerWindowClear(TfEditCookie ec, ITfContext *pContext)
{
	// ʃEChE̕NA
	auto prc = cursorManager_.GetPRC(ec, _pComposition);
	wndInputText->SetText(L"", prc);
}
