//
// compose.cpp
//
// Composition code.
//

#include "Log.h"
#include "globals.h"
#include "mark.h"
#include "editsess.h"
#include "FolderUtil.h"
#include "uwp_util.h"
#include "CompositionEditSession.h"
#include "TerminateCompositionEditSession.h"

//+---------------------------------------------------------------------------
//
// _TerminateCompositionInContext
//
//----------------------------------------------------------------------------

void CMarkTextService::_TerminateCompositionInContext(ITfContext *pContext)
{
    CTerminateCompositionEditSession *pEditSession;
    HRESULT hr;

    if (pEditSession = new CTerminateCompositionEditSession(this, pContext))
    {
        pContext->RequestEditSession(_tfClientId, pEditSession, TF_ES_SYNC | TF_ES_READWRITE, &hr);
        pEditSession->Release();
    }
}

//+---------------------------------------------------------------------------
//
// SetOnComposition
//
// Callback for the "Start/End Composition" menu item.
// If we have a composition, end it.  Otherwise start a new composition over
// the selection of the current focus context.
//----------------------------------------------------------------------------

void CMarkTextService::SetComposition(ITfContext *pContext)
{
	ITfDocumentMgr *pFocusDoc;
    if (pContext->GetDocumentMgr(&pFocusDoc) != S_OK) {
	    LogOutput(__FUNCTION__, __LINE__, "Failed to getDocumentMgr");
        return;
    }

    if (!_InitTextEditSink(pFocusDoc)) {
	    LogOutput(__FUNCTION__, __LINE__, "Failed to _InitTextEditSink");
        pFocusDoc->Release();
        return;
    }
    if (!_InitLayoutSink(pFocusDoc)) {
	    LogOutput(__FUNCTION__, __LINE__, "Failed to _InitLayoutSink");
        pFocusDoc->Release();
        return;
    }

	LogOutput(__FUNCTION__, __LINE__, "Convert Start!!");

    CCompositionEditSession *pCompositionEditSession;
    HRESULT hr;

    if (pCompositionEditSession = new CCompositionEditSession(pContext, this))
    {
        // we need a document write lock
        // the CCompositionEditSession will do all the work when the
        // CCompositionEditSession::DoEditSession method is called by the context
        pContext->RequestEditSession(this->_tfClientId, pCompositionEditSession, TF_ES_READWRITE | TF_ES_ASYNCDONTCARE, &hr);

        pCompositionEditSession->Release();
    }

    pFocusDoc->Release();    
}

//+---------------------------------------------------------------------------
//
// OnCompositionTerminated
//
// Callback for ITfCompositionSink.  The system calls this method whenever
// someone other than this service ends a composition.
//----------------------------------------------------------------------------

STDAPI CMarkTextService::OnCompositionTerminated(TfEditCookie ecWrite, ITfComposition *pComposition)
{
    // we already have the composition cached, so we can ignore pComposition...
	ITfRange *pRangeComposition;
	_pComposition->GetRange(&pRangeComposition);
	if (pRangeComposition == NULL) {
		return S_FALSE;
	}

	// ͂Zbg
    auto prc = cursorManager_.GetPRC(ecWrite, _pComposition);
	wndInputText->SetText(L"", prc);
	wndInputText->Show(FALSE);
    cursorManager_.ResetPRC(true);
    compositor_.Reset();
    pRangeComposition->Release();

    // all this service wants to do is clear the display property
    _ClearCompositionDisplayAttributes(ecWrite);

    // releae our cached composition
    SafeReleaseClear(_pComposition);
	LogOutput(__FUNCTION__, __LINE__, "Terminated");

    return S_OK;
}

//+---------------------------------------------------------------------------
//
// _ClearCompositionDisplayAttributes
//
//----------------------------------------------------------------------------

void CMarkTextService::_ClearCompositionDisplayAttributes(TfEditCookie ec)
{
    ITfRange *pRangeComposition;
    ITfContext *pContext;
    ITfProperty *pDisplayAttributeProperty;

    // we need a range and the context it lives in
    if (_pComposition->GetRange(&pRangeComposition) != S_OK)
        return;

    if (pRangeComposition->GetContext(&pContext) != S_OK)
    {
        pContext = NULL;
        goto Exit;
    }

    // get our the display attribute property
    if (pContext->GetProperty(GUID_PROP_ATTRIBUTE, &pDisplayAttributeProperty) != S_OK)
        goto Exit;

    // clear the value over the range
    pDisplayAttributeProperty->Clear(ec, pRangeComposition);

    pDisplayAttributeProperty->Release();

Exit:
    pRangeComposition->Release();
    SafeRelease(pContext);
}

//+---------------------------------------------------------------------------
//
// _SetCompositionDisplayAttributes
//
//----------------------------------------------------------------------------

BOOL CMarkTextService::_SetCompositionDisplayAttributes(TfEditCookie ec)
{
    ITfRange *pRangeComposition;
    ITfContext *pContext;
    ITfProperty *pDisplayAttributeProperty;
    VARIANT var;
    HRESULT hr;

    // we need a range and the context it lives in
    if (_pComposition->GetRange(&pRangeComposition) != S_OK)
        return FALSE;

    hr = E_FAIL;

    if (pRangeComposition->GetContext(&pContext) != S_OK)
    {
        pContext = NULL;
        goto Exit;
    }

    // get our the display attribute property
    if (pContext->GetProperty(GUID_PROP_ATTRIBUTE, &pDisplayAttributeProperty) != S_OK)
        goto Exit;

    // set the value over the range
    // the application will use this guid atom to lookup the acutal rendering information
    var.vt = VT_I4; // we're going to set a TfGuidAtom
    var.lVal = _gaDisplayAttribute; // our cached guid atom for c_guidMarkDisplayAttribute

    hr = pDisplayAttributeProperty->SetValue(ec, pRangeComposition, &var);

    pDisplayAttributeProperty->Release();

Exit:
    pRangeComposition->Release();
    SafeRelease(pContext);
    return (hr == S_OK);
}

//+---------------------------------------------------------------------------
//
// _InitDisplayAttributeGuidAtom
//
// Because it's expensive to map our display attribute GUID to a TSF
// TfGuidAtom, we do it once when Activate is called.
//----------------------------------------------------------------------------

BOOL CMarkTextService::_InitDisplayAttributeGuidAtom()
{
    ITfCategoryMgr *pCategoryMgr;
    HRESULT hr;

    if (CoCreateInstance(CLSID_TF_CategoryMgr,
                         NULL, 
                         CLSCTX_INPROC_SERVER, 
                         IID_ITfCategoryMgr, 
                         (void**)&pCategoryMgr) != S_OK)
    {
        return FALSE;
    }

    hr = pCategoryMgr->RegisterGUID(c_guidMarkDisplayAttribute, &_gaDisplayAttribute);

    pCategoryMgr->Release();
        
    return (hr == S_OK);
}
