#pragma once
#include <wtypes.h>

namespace KeyboardUtils {
	class Composer
	{
	public:
		Composer(BOOL with_shift) : with_shift(with_shift){}

		WCHAR operator()(WCHAR normal, WCHAR shifted) {
			// Caps LockON
			if (GetKeyState(VK_CAPITAL) & 0x0001) {
				// SHIFTL[ON
				if (with_shift) {
					return normal;
				}
				// SHIFTL[OFF
				else {
					return shifted;
				}
			}
			// Caps LockOFF
			else
			{
				// SHIFTL[ON
				if (with_shift) {
					return shifted;
				}
				// SHIFTL[OFF
				else {
					return normal;
				}
			}
		}

	private:
		BOOL with_shift;
	};

	class KeyboardManager
	{
	public:
		virtual BOOL Available(WPARAM wParam) = 0;
		virtual WCHAR ComposeChar(WPARAM wParam, BOOL with_shift) = 0;

	protected:
		BOOL IsAlph(WPARAM wParam) {
			return (0x41 <= wParam) && (wParam <= 0x5A);
		}

		BOOL IsDigit(WPARAM wParam) {
			return 0x30 <= wParam && (wParam <= 0x39);
		}
	};

	class JpKeyboardManager : public KeyboardManager {
	public:
		BOOL Available(WPARAM wParam) {
			if (IsDigit(wParam) || IsAlph(wParam)) {
				return TRUE;
			}

			switch (wParam) {
			case 0xBD: // -
			case 0xBB: // ^
			case 0xFF: // en mark
			case 0xDB: // @
			case 0xDD: // [
			case 0xBA: // ;
			case 0xDE: // :
			case 0xDC: // ]
			case 0xBC: // ,
			case 0xBE: // .
			case 0xBF: // /
				return TRUE;
			default:
				return FALSE;
			}
		}

		WCHAR ComposeChar(WPARAM wParam, BOOL with_shift) {
			Composer compose(with_shift);
			if (IsAlph(wParam)) {
				return ComposeAlph(wParam, compose);
			} 
			if (IsDigit(wParam)) {
				return ComposeDigit(wParam, compose);
			}

			// zL[R[hASCIIR[hɕϊ
			switch (wParam) {
			case 0xBD: // -
				return compose(0x2D, 0x3D);
			case 0xBB: // ^
				return compose(0x5E, 0x7E);
			case 0xFF: // en mark
				return compose(0x5C, 0x7C);
			case 0xDB: // @
				return compose(0x40, 0x60);
			case 0xDD: // [
				return compose(0x5B, 0x7B);
			case 0xBA: // ;
				return compose(0x3B, 0x2B);
			case 0xDE: // :
				return compose(0x3A, 0x2A);
			case 0xDC: // ]
				return compose(0x5D, 0x7D);
			case 0xBC: // ,
				return compose(0x2C, 0x3C);
			case 0xBE: // .
				return compose(0x2E, 0x3E);
			case 0xBF: // /
				return compose(0x2F, 0x3F);
			default:
				// Ή̕
				return 0x00;
			}
		}

	private:
		WCHAR ComposeAlph(WPARAM wParam, Composer &compose) {
			// At@xbgE͈̔͂ł͕R[hƈv邽char^ɑ\
			// https://docs.microsoft.com/en-us/windows/win32/inputdev/virtual-key-codes
			return compose((WCHAR)wParam | 32, (WCHAR)wParam);
		}

		WCHAR ComposeDigit(WPARAM wParam, Composer &compose) {
			if (wParam == 0x30) {
				return compose((WCHAR)wParam, 0x00);
			}
			// QWERTYz̏ꍇ̓VtgƂ͕̕R[hƏԂvĂ
			return compose((WCHAR)wParam, (WCHAR)wParam - 16);
		}
	};

	class UsKeyboardManager : public KeyboardManager {
	public:
		BOOL Available(WPARAM wParam) {
			if (IsDigit(wParam) || IsAlph(wParam)) {
				return TRUE;
			}

			switch (wParam) {
			case 0xBD: // -
			case 0xBB: // =
			case 0xDB: // [
			case 0xDD: // ]
			case 0xDC: // backslash
			case 0xBA: // ;
			case 0xDE: // '
			case 0xBC: // ,
			case 0xBE: // .
			case 0xBF: // /
			case 0xC0: // `
				return TRUE;
			default:
				return FALSE;
			}
		}

		WCHAR ComposeChar(WPARAM wParam, BOOL with_shift) {
			Composer compose(with_shift);
			if (IsAlph(wParam)) {
				return ComposeAlph(wParam, compose);
			} 
			if (IsDigit(wParam)) {
				return ComposeDigit(wParam, compose);
			}

			// zL[R[hASCIIR[hɕϊ
			switch (wParam) {

			case 0xBD: // -
				return compose(0x2D, 0x5F);
			case 0xBB: // =
				return compose(0x3D, 0x2B);
			case 0xDC: // backslash
				return compose(0x5C, 0x7C);
			case 0xDB: // [
				return compose(0x5B, 0x7B);
			case 0xDD: // ]
				return compose(0x5D, 0x7D);
			case 0xBA: // ;
				return compose(0x3B, 0x3A);
			case 0xDE: // '
				return compose(0x27, 0x22);
			case 0xBC: // ,
				return compose(0x2C, 0x3C);
			case 0xBE: // .
				return compose(0x2E, 0x3E);
			case 0xBF: // /
				return compose(0x2F, 0x3F);
			case 0xC0: // `
				return compose(0x60, 0x7E);
			default:
				// Ή̕
				return 0x00;
			}
		}

	private:
		WCHAR ComposeAlph(WPARAM wParam, Composer &compose) {
			// At@xbgE͈̔͂ł͕R[hƈv邽char^ɑ\
			// https://docs.microsoft.com/en-us/windows/win32/inputdev/virtual-key-codes
			WCHAR orig = (WCHAR) wParam;
			return compose((WCHAR)wParam | 32, (WCHAR)wParam);
		}


		WCHAR ComposeDigit(WPARAM wParam, Composer &compose) {
			WCHAR c;
			if (wParam == 0x30) {
				c = L')';
			}
			else {
				c = L"!@#$%^&*("[wParam - '1'];
			}
			return compose((WCHAR)wParam, c);
		}
	};
}
