﻿#include "AkkharaConversionTable.h"
#include "FolderUtil.h"


///////////////////////////////////////////////////////////
// CAkkharaConversionDataクラス
// AKKHARA変換処理情報クラス
///////////////////////////////////////////////////////////
// コンストラクタ
CAkkharaConversionData::CAkkharaConversionData()
{
	beforeWordsShow.clear();
	afterWordsShow.clear();
	beforeWords.clear();
	afterWords.clear();
	order = 0;
	use_flag = 0;
}

// デストラクタ
CAkkharaConversionData::~CAkkharaConversionData()
{
	beforeWordsShow.clear();
	afterWordsShow.clear();
	beforeWords.clear();
	afterWords.clear();
}

// データをセットする
BOOL CAkkharaConversionData::SetData(wchar_t* buffer)
{
	std::vector<std::wstring> vecwstrColumn;

	// 改行コードを除去
	buffer[wcslen(buffer) - 1] = '\0';
	// 文字列を指定した区切り文字列で分割する
	split(buffer, L" ||| ", vecwstrColumn);

	// カラム数が4個以外の場合はフォーマットエラーとみなす
	if (vecwstrColumn.size() != 4) {
		return FALSE;
	}

	// 変換前文字列
	std::vector<std::wstring> wordTemp;
	// 文字列を指定した区切り文字列で分割する
	split(vecwstrColumn[ColumnKind::BEFORE], L" ", wordTemp);
	// 変換前文字列（表示用）
	beforeWordsShow = wordTemp;
	// 変換前文字列（変換用）
	beforeWords = wordTemp;
	for (size_t i = 0; i < beforeWords.size(); i++) {
		short sVal = (short)wcstol(beforeWords[i].c_str(), NULL, 16);
		beforeWords[i] = sVal;
	}

	// 変換後文字列
	wordTemp.clear();
	// 文字列を指定した区切り文字列で分割する
	split(vecwstrColumn[ColumnKind::AFTER], L" ", wordTemp);
	// 変換後文字列（表示用）
	afterWordsShow = wordTemp;
	// 変換後文字列（変換用）
	afterWords = wordTemp;
	for (size_t i = 0; i < afterWords.size(); i++) {
		short sVal = (short)wcstol(afterWords[i].c_str(), NULL, 16);
		afterWords[i] = sVal;
	}

	// 変換順序
	// 文字列を指定した区切り文字列で分割する
	order = stoi(vecwstrColumn[ColumnKind::ORDER]);
	// 処理順序が0の時はキー入力とみなし対象のキーコードの値を取得する
	if (order == 0) {
		// キー入力文字
		std::string strKey = WStringToString(vecwstrColumn[ColumnKind::BEFORE]);
		keyCode = strKey[0];
	}

	// 利用者選択可/不可のフラグ
	// 文字列を指定した区切り文字列で分割する
	use_flag = stoi(vecwstrColumn[ColumnKind::USE_FLAG]);

	return TRUE;
}

// キー入力文字を取得する
char CAkkharaConversionData::GetKeyCode()
{
	return keyCode;
}

// 変換前文字列（表示用）を取得する
std::vector<std::wstring> CAkkharaConversionData::GetBeforeWordsShow()
{
	return beforeWordsShow;
}

// 変換前文字列（変換用）を取得する
std::vector<std::wstring> CAkkharaConversionData::GetBeforeWords()
{
	return beforeWords;
}

// 変換後文字列（表示用）を取得する
std::vector<std::wstring> CAkkharaConversionData::GetAfterWordsShow()
{
	return afterWordsShow;
}

// 変換後文字列（変換用）を取得する
std::vector<std::wstring> CAkkharaConversionData::GetAfterWords()
{
	return afterWords;
}

// 変換順序を取得する
int CAkkharaConversionData::GetOrder()
{
	return order;
}

// 利用者選択可/不可のフラグを取得する
int CAkkharaConversionData::GetUseFlag()
{
	return use_flag;
}

// 利用者選択可/不可のフラグをセットする
void CAkkharaConversionData::SetUseFlag(int flag)
{
	use_flag = flag;
}

// デバッグ用ログ出力
void CAkkharaConversionData::DebugLog()
{
	std::wstring wsData = L"";
	wsData = L"before[ ";
	for (size_t i = 0; i < beforeWordsShow.size(); i++) {
		wsData += beforeWordsShow[i] + L" ";
	}
	wsData += L"] after[ ";
	for (size_t i = 0; i < afterWordsShow.size(); i++) {
		wsData += afterWordsShow[i] + L" ";
	}
	wsData += L"]";

	if (order == 0) {
		LogOutputW(__FUNCTION__, __LINE__, L"keyCode[%c][0x%x] %ls order[%d] use_flag[%d]", keyCode, keyCode, wsData.c_str(), order, use_flag);
	}
	else {
		LogOutputW(__FUNCTION__, __LINE__, L"%ls order[%d] use_flag[%d]", wsData.c_str(), order, use_flag);
	}
}


///////////////////////////////////////////////////////////
// CAkkharaConversionOrderクラス
// 処理順序毎のAKKHARA変換処理情報クラスリストを保持する
///////////////////////////////////////////////////////////
// コンストラクタ
CAkkharaConversionOrder::CAkkharaConversionOrder()
{
	conversion.clear();
}

// デストラクタ
CAkkharaConversionOrder::~CAkkharaConversionOrder()
{
}

// AKKHARA変換処理情報クラスを追加する
void CAkkharaConversionOrder::AddConversion(CAkkharaConversionData akkConv)
{
	conversion.push_back(akkConv);
}

// AKKHARA変換処理情報クラスリスト数を取得
size_t CAkkharaConversionOrder::GetConversionNum()
{
	return conversion.size();
}

// AKKHARA変換処理情報クラスを取得
CAkkharaConversionData CAkkharaConversionOrder::GetConversion(int index)
{
	return conversion[index];
}

// 利用者選択可/不可のフラグを更新する
void CAkkharaConversionOrder::UpdateUseFlag(int index, int flag)
{
	CAkkharaConversionData conv = GetConversion(index);
	conv.SetUseFlag(flag);
	conversion[index] = conv;
}


///////////////////////////////////////////////////////////
// CAkkharaConversionTableクラス
// AKKHARA変換テーブルクラス
///////////////////////////////////////////////////////////
// コンストラクタ
CAkkharaConversionTable::CAkkharaConversionTable()
{
	keyInput.clear();
	conversionOrder.clear();
	normalization.clear();
	m_nCurSelected = 0;
}

// デストラクタ
CAkkharaConversionTable::~CAkkharaConversionTable()
{
}

// AKKHARAファイルを検索する
BOOL CAkkharaConversionTable::SeekAkkharaFile()
{
	vecwstrLanguageList.clear();
	vecwstrAkkharaFilePath.clear();

	// ユーザーフォルダ内のAKKHARAフォルダのパスを取得
	TCHAR tcsUserAkkharaDir[INSTALL_FOLDER_PATH_MAX_LENGTH];
	GetInstallFolderPath(tcsUserAkkharaDir);
	std::wstring wstrAkkharaDir = tcsUserAkkharaDir;

	LogOutputW(__FUNCTION__, __LINE__, L"strAkkharaDir [%s]", wstrAkkharaDir.c_str());
	std::wstring wstrAkkharaSeek = wstrAkkharaDir + L"\\*.akkhara";
	LogOutputW(__FUNCTION__, __LINE__, L"wstrAkkharaSeek [%s]", wstrAkkharaSeek.c_str());

	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;
	BOOL bStat = FALSE;

	// AKKHARAフォルダ内を検索
	hFind = FindFirstFile(wstrAkkharaSeek.c_str(), &FindFileData);
	if (hFind != INVALID_HANDLE_VALUE) {
		do {
			vecwstrAkkharaFilePath.push_back(wstrAkkharaDir + L"\\" + FindFileData.cFileName);
			std::vector<std::wstring> wordTemp;
			// 文字列を指定した区切り文字列で分割する
			split(FindFileData.cFileName, L".", wordTemp);
			vecwstrLanguageList.push_back(wordTemp[0]);
			LogOutputW(__FUNCTION__, __LINE__, L"FindFileData.cFileName [%s]", wordTemp[0].c_str());
			bStat = TRUE;
		} while (FindNextFile(hFind, &FindFileData));
		FindClose(hFind);
	}

	for (size_t i = 0; i < vecwstrAkkharaFilePath.size(); i++) {
		LogOutputW(__FUNCTION__, __LINE__, L"wstrAkkharaFilePath [%s]", vecwstrAkkharaFilePath[i].c_str());
	}

	return bStat;
}

// 言語リストを取得する
std::vector<std::wstring> CAkkharaConversionTable::GetLanguageList() const
{
	return vecwstrLanguageList;
}

// カレントの言語名を取得する
std::wstring CAkkharaConversionTable::GetCurrentLanguageName() const
{
	LogOutputW(__FUNCTION__, __LINE__, L"wstrAkkharaFilePath [%d][%s]", m_nCurSelected, vecwstrLanguageList[m_nCurSelected].c_str());
	return vecwstrLanguageList[m_nCurSelected];
}

// 変換テーブルの生成
BOOL CAkkharaConversionTable::CreateConvertTable(int index)
{
	keyInput.clear();
	conversionOrder.clear();
	normalization.clear();
	m_nCurSelected = index;

	LogOutputW(__FUNCTION__, __LINE__, L"vecwstrAkkharaFilePath[%d][%s]", index, vecwstrAkkharaFilePath[m_nCurSelected].c_str());
	FILE* fp = _wfopen(vecwstrAkkharaFilePath[m_nCurSelected].c_str(), TEXT("rt"));
	if (fp == NULL) {
		LogOutput(__FUNCTION__, __LINE__, "AKKHARA file open error.");
		return FALSE;
	}

	BOOL status = TRUE;
	wchar_t wsBuf[1024];
	int orderCurrent = 0;

	memset(wsBuf, 0, sizeof(wchar_t) * 1024);
	while (fgetws(wsBuf, 1024, fp) != NULL) {
		CAkkharaConversionData akkConv;
		if (akkConv.SetData(wsBuf) == FALSE) {
			status = FALSE;
			goto error;
		}

		// キー入力
		if (akkConv.GetOrder() == 0) {
			keyInput.push_back(akkConv);
		}
		// 正規化処理
		else if (akkConv.GetOrder() == -1) {
			normalization.push_back(akkConv);
		}
		// 変換処理
		else {
			if (akkConv.GetOrder() > (int)conversionOrder.size()) {
				CAkkharaConversionOrder akkharaConversionOrder;
				conversionOrder.push_back(akkharaConversionOrder);
			}
			conversionOrder[akkConv.GetOrder() - 1].AddConversion(akkConv);
		}
	}

error:
	fclose(fp);

	// デバッグ用ログ出力
//	DebugLog();

	return status;
}

// AKKHARAファイルのフォーマット文字列を取得
std::wstring CAkkharaConversionTable::GetAkkharaStr(CAkkharaConversionData akkhara)
{
	wchar_t wcOrderUseFlag[64];
	std::wstring wsData = L"";
	std::vector<std::wstring> beforeWordsShow = akkhara.GetBeforeWordsShow();
	for (size_t i = 0; i < beforeWordsShow.size(); i++) {
		wsData += beforeWordsShow[i] + L" ";
	}
	wsData += L"||| ";
	std::vector<std::wstring> afterWordsShow = akkhara.GetAfterWordsShow();
	for (size_t i = 0; i < afterWordsShow.size(); i++) {
		wsData += afterWordsShow[i] + L" ";
	}
	wsData += L"||| ";

	swprintf(wcOrderUseFlag, L"%d ||| %d", akkhara.GetOrder(), akkhara.GetUseFlag());
	wsData += wcOrderUseFlag;

	return wsData;
}

// AKKHARAファイルの書き込み
BOOL CAkkharaConversionTable::WriteAkkharaFile()
{
	BOOL status = TRUE;
	LogOutput(__FUNCTION__, __LINE__, "path [%s]", vecwstrAkkharaFilePath[m_nCurSelected].c_str());
	FILE* fp = _wfopen(vecwstrAkkharaFilePath[m_nCurSelected].c_str(), TEXT("wt"));
	if (fp == NULL) {
		LogOutput(__FUNCTION__, __LINE__, "AKKHARA file open error.");
		return FALSE;
	}

	for (size_t i = 0; i < GetKeyInputNum(); i++) {
		CAkkharaConversionData akkhara = GetKeyInput(i);
		fwprintf(fp, L"%ls\n", GetAkkharaStr(akkhara).c_str());
	}
	for (size_t i = 0; i < GetConversionOrdersNum(); i++) {
		for (size_t j = 0; j < GetConversionNum(i); j++) {
			CAkkharaConversionData akkhara = GetConversion(i, j);
			fwprintf(fp, L"%ls\n", GetAkkharaStr(akkhara).c_str());
		}
	}
	for (size_t i = 0; i < GetNormalizationNum(); i++) {
		CAkkharaConversionData akkhara = GetNormalization(i);
		fwprintf(fp, L"%ls\n", GetAkkharaStr(akkhara).c_str());
	}

	fclose(fp);

	return status;
}

// キー入力のAKKHARA変換処理情報クラスリスト数を取得
size_t CAkkharaConversionTable::GetKeyInputNum()
{
	return keyInput.size();
}

// 処理順序リスト数を取得
size_t CAkkharaConversionTable::GetConversionOrdersNum()
{
	return conversionOrder.size();
}

// 処理順序毎のAKKHARA変換処理情報クラスリスト数を取得
size_t CAkkharaConversionTable::GetConversionNum(int order)
{
	return conversionOrder[order].GetConversionNum();
}

// 正規化処理のAKKHARA変換処理情報クラスリスト数を取得
size_t CAkkharaConversionTable::GetNormalizationNum()
{
	return normalization.size();
}

// キー入力のAKKHARA変換処理情報クラスを取得
CAkkharaConversionData CAkkharaConversionTable::GetKeyInput(int index)
{
	return keyInput[index];
}

// 指定されたcharに対する文字がKeyInputに存在するか調べ、存在する場合は該当のConversionDataを返す
CAkkharaConversionData *CAkkharaConversionTable::LookupKeyInput(const char c) {
	/*
	TODO:
	このような場合に線形探索を使うべきでなく、keyInputのデータ構造としてstd::unordered_mapなどを使うべきである。
	しかし、初めに開発した会社が線形探索に依存したコードを書いてしまっているためすぐには書き換えられないため引き継ぎ事項とする。
	*/
	for (auto &ki : keyInput) {
		char keyCode = ki.GetKeyCode();
		if (keyCode == c) {
			return &ki;
		}
	}
	return NULL;
}

// 処理順序毎のAKKHARA変換処理情報クラスを取得
CAkkharaConversionData CAkkharaConversionTable::GetConversion(int order, int index)
{
	return conversionOrder[order].GetConversion(index);
}

// 正規化処理のAKKHARA変換処理情報クラスを取得
CAkkharaConversionData CAkkharaConversionTable::GetNormalization(int index)
{
	return normalization[index];
}

// 利用者選択可/不可のフラグを更新する
void CAkkharaConversionTable::UpdateUseFlag(int order, int index, int flag)
{
	conversionOrder[order].UpdateUseFlag(index, flag);
}

// デバッグ用ログ出力
void CAkkharaConversionTable::DebugLog()
{
	LogOutputW(__FUNCTION__, __LINE__, L"keyInput");
	for (size_t i = 0; i < GetKeyInputNum(); i++) {
		CAkkharaConversionData akkhara = GetKeyInput(i);
		akkhara.DebugLog();
	}
	LogOutputW(__FUNCTION__, __LINE__, L"conversion");
	for (size_t i = 0; i < GetConversionOrdersNum(); i++) {
		LogOutputW(__FUNCTION__, __LINE__, L"order [%d]", i + 1);
		for (size_t j = 0; j < GetConversionNum(i); j++) {
			CAkkharaConversionData akkhara = GetConversion(i, j);
			akkhara.DebugLog();
		}
	}
	LogOutputW(__FUNCTION__, __LINE__, L"normalization");
	for (size_t i = 0; i < GetNormalizationNum(); i++) {
		CAkkharaConversionData akkhara = GetNormalization(i);
		akkhara.DebugLog();
	}
}
